function y = fdpGeneralN(x,tipoFDP);
%
% y = fdpGeneralN(x,tFDP);
%
% Function that computes the probability density function (PDF) defined by 
% the parameters in vector tFDP=[type,deltaS,centerS]
%
%     x : input argument for the evaluation of the PDF (1 x Nx)
%  tFDP : parameters of the PDF
%
%     tFDP(1) defines the type of the PDF 
%      >  0 : Gaussian PDF with mean centerS 
%             (with 99% of probability in the interval centerS +/- deltaS)
%      >  1 : Uniform PDF in centerS +/- deltaS  
%             (Equivalent to L1 norm in +/-1 when used to train MLPs)
%      >  2 : Linear increasing PDF in centerS +/- deltaS 
%             (Equivalent to L2 norm in +/-1 when used to train MLPs)
%      > -2 : Linear decreasing PDF in centerS +/- deltaS
%             (Equivalent to complementary of L2 norm in +/-1 when used to train MLPs)
%      >  3 : Quadratic increasing PDF in centerS +/- deltaS 
%             (Equivalent to L3 norm in +/-1 when used to train MLPs)
%      > -3 : Quadratic decreasing PDF in centerS +/- deltaS
%             (Equivalent to complementary of L3 norm in +/-1 when used to train MLPs)
%      >  4 : Cubic increasing PDF in centerS +/- deltaS 
%             (Equivalent to L4 norm in +/-1 when used to train MLPs)
%      > -4 : Cubic decreasing PDF in centerS +/- deltaS 
%             (Equivalent to complementary of L4 norm in +/-1 when used to train MLPs)
%      >  5 : Fourth order increasing PDF in centerS +/- deltaS 
%             (Equivalent to L5 norm in +/-1 when used to train MLPs)
%      > -5 : Fourth order decreasing PDF in centerS +/- deltaS
%             (Equivalent to complementary L5 norm in +/-1 when used to train MLPs)
%
%      > 10 : Triangle PDF symmetric in centerS +/- deltaS
%      > 11 : Absolute value PDF in centerS +/- deltaS
%      > 12 : Integral of the tanh PDF function 
%           (con 99% de probabilidad en centroS +/- deltaS)
%      > 13 : Exponential PDF with margin 
%            (margin given by centerS, and decay factor deltaS)
%      >-13 : Complementary exponential PDF with margin 
%            (margin given by centerS, and decay factor deltaS)
%
%      > 102, -102, 110, 111: generalizations of 2,-2,10,11 with sinusoidal
%             transitions, instead of linear transitions
%
%      > 202, -202, 210, 211: generalizations of 2,-2,10,11 with quadratic
%             transitions (x^2), instead of linear transitions
%
%      > 302, -302, 310, 311: generalizations de 2,-2,10,11 with inverted
%             quadratic transitions (1 - x^2), instead of linear transitions
%
%
%     tFDP(2) provides parameter deltaS
%     tFDP(3) provides parameter centerS
%
%--------------------------------------------------------------------------
%        Author: Marcelino Lzaro
%      Creation: December 2014
%   Last update: April 2019
%--------------------------------------------------------------------------


% y = fdpGeneralN(x,tFDP);
%
% Funcin que calcula la funcin densidad de probabilidad especificada
% y parmetro tFDP=[tipo,deltaS,centroS]
%
%  tFDP : parametros FDP base para el estimador de Parzen
%
%     tFDP(1) indica el tipo de FDP 
%      >  0 : FDP Gausiana (99% de probabilidad en centroS +/- deltaS)
%      >  1 : FDP Uniforme en centroS +/- deltaS  (norma L1 en +/-1)
%      >  2 : FDP Lineal Creciente en centroS +/- deltaS (norma L2 en +/-1)
%      > -2 : FDP Lineal Decreciente en centroS +/- deltaS
%      >  3 : FDP cuadrtica Creciente en centroS +/- deltaS (norma L3 en +/-1)
%      > -3 : FDP cuadrtica Decreciente en centroS +/- deltaS (complementp norma L3 en +/-1)
%      >  4 : FDP cbica Creciente en centroS +/- deltaS (norma L4 en +/-1)
%      > -4 : FDP cbica Decreciente en centroS +/- deltaS (complementp norma L4 en +/-1)
%      >  5 : FDP cuarta Creciente en centroS +/- deltaS (norma L5 en +/-1)
%      > -5 : FDP cuarta Decreciente en centroS +/- deltaS (complementp norma L5 en +/-1)
%
%      > 10 : FDP Triangular simtrica en centroS +/- deltaS
%      > 11 : FDP Valor Absoluto en centroS +/- deltaS
%      > 12 : FDP Integral de activacin tanh 
%           (con 99% de probabilidad en centroS +/- deltaS)
%      > 13 : FDP exponencial con margen (factor deltaS, margen centroS)
%      >-13 : FDP exponencial con margen invertida (factor deltaS, margen centroS)
%
%      > 102, -102, 110, 111: generalizaciones de 2,-2,10,11 pero con
%           transiciones sinusoidales en lugar de lineales
%
%      > 202, -202, 210, 211: generalizaciones de 2,-2,10,11 pero con
%           transiciones exponenciales (x^2) en lugar de lineales
%
%      > 302, -302, 310, 311: generalizaciones de 2,-2,10,11 pero con
%           transiciones exponenciales inversas (1 - x^2) 
%
%
%     tFDP(2) indica el parmetro de ancho deltaS (soporte centroS +/- deltaS)
%     tFDP(3) indica el parmetro de media (centroS)
%
%--------------------------------------------------------------------------
%         Autor: Marcelino Lzaro
%      Creacin: diciembre 2014
% Actualizacin: abril 2017
%--------------------------------------------------------------------------
%

tFDP=tipoFDP(1);
deltaS=tipoFDP(2);
if length(tipoFDP) >= 3
    centroS=tipoFDP(3);
else
    centroS=0;
end

if tFDP == 0 
    %adapta=3.09022; % 99.8%
    %adapta=2.32635; % 98%
    adapta=2.5758; % 99%
    y = fdpGaussiana(x-centroS,deltaS/adapta);
elseif tFDP == 1 
    y = fdpUniforme(x-centroS,deltaS);
elseif tFDP == 2
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v)=(x(v)-centroS+deltaS)/deltaS/deltaS/2;    
elseif tFDP == -2
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v)=(-x(v)+centroS+deltaS)/deltaS/deltaS/2;
elseif tFDP == 3
    y=zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 3/8/deltaS*(1+(x(v)-centroS)/deltaS).^2;
elseif tFDP == -3
    y=zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 3/8/deltaS*(1-(x(v)-centroS)/deltaS).^2;    
elseif tFDP == 4
    y=zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 1/4/deltaS*(1+(x(v)-centroS)/deltaS).^3;
elseif tFDP == -4
    y=zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 1/4/deltaS*(1-(x(v)-centroS)/deltaS).^3;    
elseif tFDP == 5
    y=zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 5/32/deltaS*(1+(x(v)-centroS)/deltaS).^4;
elseif tFDP == -5
    y=zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 5/32/deltaS*(1-(x(v)-centroS)/deltaS).^4;

elseif tFDP == 10 
    y = fdpTriangular(x-centroS,deltaS);
elseif tFDP == 11 
    y = fdpAbs(x-centroS,deltaS);
elseif tFDP == 12  % tanh(ax)
    adapta = 2.6467; % 99%
    y = adapta/deltaS*(1-(tanh(adapta/deltaS*(x-centroS)).^2))/2;
elseif tFDP == 13 % exp con margen
    y=zeros(size(x));
    v = find(x>=-centroS);
    y(v) = deltaS*exp(-deltaS*(x(v)+centroS));
elseif tFDP == -13% expo con margen (invertida)
    y=zeros(size(x));
    v=find(x<=centroS);
    y(v)=deltaS*exp(deltaS*(x(v)-centroS));
   
    
%------------------------------------------
% Cosenoidales
%------------------------------------------
elseif tFDP == 100
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v)=(1+cos((x(v)-centroS)*pi/deltaS))/(2*deltaS);
elseif tFDP == 110 
    y = fdpCoseno(x-centroS,deltaS);
elseif tFDP == 102
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = (1 + sin(pi/2*(x(v)-centroS)/deltaS) )/2/deltaS;
elseif tFDP == -102
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = (1 - sin(pi/2*(x(v)-centroS)/deltaS) )/2/deltaS;    
elseif tFDP == 111 
    y = fdpCosenoInv(x-centroS,deltaS);
    
%------------------------------------------
% Exponenciales
%------------------------------------------
elseif tFDP == 210
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = ( (abs(x(v)-centroS)-deltaS).^2 )*(3/2/deltaS^3);
elseif tFDP == 202
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = ( (x(v)-centroS+deltaS).^2 )*(3/8/deltaS^3);
elseif tFDP == -202
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = ( (x(v)-centroS-deltaS).^2 )*(3/8/deltaS^3);
elseif tFDP == 211
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = ( (x(v)-centroS).^2 )*(3/2/deltaS^3);

%------------------------------------------
% Exponenciales invertidas
%------------------------------------------
elseif tFDP == 310
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 3/4/deltaS - ( (x(v)-centroS).^2 )*(3/4/deltaS^3);
elseif tFDP == 302
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 3/4/deltaS - ( (x(v)-centroS-deltaS).^2 )*(3/16/deltaS^3);
elseif tFDP == -302
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 3/4/deltaS - ( (x(v)-centroS+deltaS).^2 )*(3/16/deltaS^3);
elseif tFDP == 311
    y = zeros(size(x));
    v = find(abs(x-centroS)<=deltaS);
    y(v) = 3/4/deltaS - ( (abs(x(v)-centroS)-deltaS).^2 )*(3/4/deltaS^3);
    
%------------------------------------------
% Algunas pruebas
%------------------------------------------
elseif tFDP == 106
    y=zeros(size(x));
    v = find( ((x-centroS)>=-deltaS) & ((x-centroS)<-deltaS/2) );
    y(v) = (x(v)-centroS+deltaS)*8/11/deltaS^2;
    v = find(abs(x-centroS)<=deltaS/2);
    y(v) = 8/11/deltaS;
    v = find( ((x-centroS)>=deltaS/2) & ((x-centroS)<deltaS) );
    y(v) = 4/11/deltaS;

end
