




/*
 *  Copyright 2003 by Spectrum Digital Incorporated.
 *  All rights reserved. Property of Spectrum Digital Incorporated.
 */

/*
 *  ======== led.c ========
 *
 *  This example blinks LED #0 at a rate of about 2.5 times per second using
 *  the LED module of the the DSK6713 Board Support Library.  The example
 *  also reads the state of DIP switch #3 and lights LED #3 if the switch
 *  is depressed or turns it off if the switch is not depressed.
 *
 *  The purpose of this example is to demonstrate basic BSL usage as well
 *  as provide a project base for your own code.
 *
 *  Please see the DSK6713 help file for more detailed information.
 */

/*
 *  DSP/BIOS is configured using the DSP/BIOS configuration tool.  Settings
 *  for this example are stored in a configuration file called led.cdb.  At
 *  compile time, Code Composer will auto-generate DSP/BIOS related files
 *  based on these settings.  A header file called ledcfg.h contains the
 *  results of the autogeneration and must be included for proper operation.
 *  The name of the file is taken from led.cdb and adding cfg.h.
 */
#include "sinecfg.h"

/*
 *  The Board Support Library is divided into several modules, each
 *  of which has its own include file.  The file dsk6713.h must be included
 *  in every program that uses the BSL.  This example also includes
 *  dsk6713_led.h and dsk6713_dip.h because it uses the LED and DIP modules.
 */
#include "dsk6713.h"
#include "dsk6713_led.h"
#include "dsk6713_dip.h"
#include "dsk6713_aic23.h"
#include <csl_irq.h>

/*
 *  main() - Main code routine, initializes BSL and runs LED application
 */

/*
 *  EXTRA: Pressing DIP switch #3 changes LED #3 from off to on.
 */

int active_led=0;
int pos_tbl_c0=0;
int pos_tbl_c1=0;
int flag=0;

/*
 * Configuracion del AC23
 *
 */

#define SINE_TABLE_SIZE 16;

DSK6713_AIC23_Config config = {
    0x0017, // 0 DSK6713_AIC23_LEFTINVOL  Left line input channel volume
    0x0017, // 1 DSK6713_AIC23_RIGHTINVOL Right line input channel volume
    0x00d8, // 2 DSK6713_AIC23_LEFTHPVOL  Left channel headphone volume
    0x00d8, // 3 DSK6713_AIC23_RIGHTHPVOL Right channel headphone volume
    0x0011, // 4 DSK6713_AIC23_ANAPATH    Analog audio path control
    0x0000, // 5 DSK6713_AIC23_DIGPATH    Digital audio path control
    0x0000, // 6 DSK6713_AIC23_POWERDOWN  Power down control
    0x0043, // 7 DSK6713_AIC23_DIGIF      Digital audio interface format
    0x0001, // 8 DSK6713_AIC23_SAMPLERATE Sample rate control
    0x0001  // 9 DSK6713_AIC23_DIGACT     Digital interface activation
};


Int16 sinetableC0[] = {0,6270,11585,15137,16384,15137,11585,6270,0,-6270,-11585,-15137,-16384,-15137,-11585,-6270};

Int16 sinetableC1[] = {0,11585,16384,11585,0,-11585,-16384,-11585,0,11585,16384,11585,0,-11585,-16384,-11585};

DSK6713_AIC23_CodecHandle hCodec;

void init_hwi();
void rotate_leds();
void initIrq();

void main()
{


    /* Initialize the board support library, must be first BSL call */
    DSK6713_init();

    /* Initialize the LED and DIP switch modules of the BSL */
    DSK6713_LED_init();
    DSK6713_DIP_init();
    
	active_led=0;
	flag=0;
    IRQ_globalDisable();       // Disable global interrupts during setup

	//CODEC  Initialization
    hCodec = DSK6713_AIC23_openCodec(0, &config);
	DSK6713_AIC23_setFreq(hCodec,DSK6713_AIC23_FREQ_8KHZ);
	initIrq();
	IRQ_globalEnable();       // Disable global interrupts during setup

}

void rotate_leds()
{
	int next_led=active_led;
	next_led=(active_led+1) % 4;
	DSK6713_LED_on(next_led);
	DSK6713_LED_off(active_led);
	active_led=next_led;
}

void initIrq()
{
    /* Enable RXTX interrupts to the CPU */
    IRQ_clear(IRQ_EVT_RINT1);   // Clear any pending RX interrupts
    IRQ_enable(IRQ_EVT_RINT1);   // Enable RX interrupt
}

void procRXInt(void)
{
  Uint32 Channel=0;
  Int16 Output=0;

  DSK6713_AIC23_read(hCodec,&Channel);
  if (! flag) {
    Output = sinetableC0[pos_tbl_c0];
    pos_tbl_c0 = (pos_tbl_c0 +1)% SINE_TABLE_SIZE;
  } else {
    Output = sinetableC1[pos_tbl_c1];
    pos_tbl_c1 = (pos_tbl_c1 + 1)% SINE_TABLE_SIZE;
  }
  flag = !flag;
    
  DSK6713_AIC23_write(hCodec,Output);
}
