%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% gm (Generative Model) that carries out and image retrieval task between
%%% one query image and R reference images Sintaxis: [similarities]=
%%% gm(query_path,reference_paths,fq,fr,matches,K) INPUTS:  - query_path -
%%% path to the Query Image
%%%          - reference_paths - 1xR cell of strings containing the paths
%%%          to the reference images. - fq - A struct with the N query
%%%          features as returned by VLFEAT: contains tow elements:
%%%             1.- fq.f is a 6xN matrix with the ellipsoids descriptors of
%%%             each detected feature in the query. In practice, it is just
%%%             required that the first 2 components are the (x,y)
%%%             coordinates of the feature. 2.- fq.d is a MxN matrix with
%%%             the M-dimensional features descriptors of each detected
%%%             feature in the query.
%%%          - fr - A 1xR cell array with the features corresponding to
%%%          each image in the reference set. The format for each
%%%          particular set is the same as fq. - matches - A 1xR cell array
%%%          with the matched features between the query and each reference
%%%          image. Each element matches{r} of the array corresponds with a
%%%          reference r and is a structure with two elements:
%%%             1.- matches{r}.idx: a 2xMr matrix. Mr is the number of
%%%             matched points between the query and the reference image r.
%%%             The first row contains the features index in the query,
%%%             whereas the second one contains the features index in the
%%%             reference. 2.- matches{r}.dists: a 1xMr vector with the
%%%             square of the euclidean distance between the matched
%%%             descriptors.
%%%          - K - The number of components in the generative model
%%%          including both the BG component and the number of FG
%%%          components.
%%%
%%% OUTPUT:  similarities - A Rx1 vector with the similarities between the
%%% query and the reference images.
%%%
%%%
%%% Multimedia Processing Group. Universidad Carlos III de Madrid.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [similarities]= gm(query_path,reference_paths,fq,fr,matches,K)

%%%%%%%%%%%%%%%%%%
%%% PARAMETERS %%%
%%%%%%%%%%%%%%%%%%

% General Parameters
niter = 30;                   % Hybrid model algorithm max iterations
stop_threshold = 1e-4;        % Threshold to stop hm algorithm
show_figures =0;              % 1: View plots online (debug)
minValue = 1e-100;            % Useful to avoid Nan or Infs
min_points_per_image = 20;    % Minimum points for an image to be considered in the Generative Model
minNumberImages = 5;                % Minimum number of valid images to run the code.
% Spatial Consistency Parameters
mult=1;	                      % Uniform (k=1) factor for mu prob (just when Gaussians are employed)
seg_SC = 0;                   % 1: Segmentation-based Spatial Consistency. 1: active   0: inactive
segPath = './segmentation/segment'; % Path to the executable of the segmentation algorithm
ext3_k = 200;                 % Segmentation Algorithm: Value for the threshold function in the segmentation
ext3_min = 1000;              % Segmentation Algorithm: Minimum component size enforced by post-processing
ext3_sigma = 1;               % Segmentation Algorithm: Used to smooth the input image before segmenting it

% Transformation-based Location Parameters
multA=0.25;                   % Uniform (k=1) factor for affine transform prob (between -1 and 1)
b_reg=10;					  % Regularization term that controls the balance between the prior and the data-based sigmaA
priorA=2e-3*eye(2);           % Prior for sigmaA matrix


% Visual Similarity Parameters
vis_similarity = 1;             % 1: Visual Similarity active   0: Visual Similarity inactive
ext_constant_lambda = 0;      % 1: lambda is initial_lambda (no adaptation)
ext_initial_lambda = 5;       % Initial value. If constant_lambda set to 1 lamba=initial_lambda;



%%%%%%%%%%%%%%%
%%%  START  %%%
%%%%%%%%%%%%%%%
warning off

%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  READING QUERY DATA %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%

Iq=imread(query_path);
locs = fq.f(1:2,:)';
x_i_q = locs;
coord_q=locs;
N=length(x_i_q);
[hc wc aux]=size(Iq);
tc=max(wc,hc);
tc=1024;

%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  SEGMENTATION STEP  %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%
if seg_SC
    [region_borders,mult,as_idx,color_asign,Iq_segmt] = segment(Iq,ext3_sigma,ext3_k,ext3_min,segPath);
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  PROCESSING POINTS  %%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Processing points...');
Nref=length(reference_paths);

nidx = 0;
x_i_q_t=[];
x_i_r_t=[];
x_i_index=[];
x_i_index2=[];
new_dists=[];
xir=cell(1,Nref);
xiq=cell(1,Nref);
num_imgs=1;

%We decide how many images are valid
validRefs=0;
while validRefs<minNumberImages 
    validRefs=0;
    for r=1:Nref
        results=matches{r}.idx;
        dists=matches{r}.dists;
        %Number of Matches
        nMatches=size(results,2);
        if(nMatches>=min_points_per_image)
            validRefs=validRefs+1;
        end
    end
    min_points_per_image=min_points_per_image-1;
    
end
min_points_per_image=min_points_per_image+1;

%Now we can allocate memory
dists_i=cell(1,validRefs);
original_dists=zeros(1,validRefs);
img_index=zeros(1,validRefs);
N_img=zeros(1,validRefs);

num_imgs=1;
for r=1:Nref
    results=matches{r}.idx;
    dists=matches{r}.dists;
    %Number of Matches
    nMatches=size(results,2);
    
    
    if(nMatches>=min_points_per_image)
        coord_r = fr{r}.f(1:2,:)';
        x_i_r = coord_r;
        %         info = imfinfo(reference_paths{r});
        
        %We generate the variables
        dists_i{num_imgs}=dists;
        xir{num_imgs}=zeros(3,nMatches);
        xir{num_imgs}(1,:) = x_i_r(results(2,:),1)./tc;
        xir{num_imgs}(2,:) = x_i_r(results(2,:),2)./tc;
        xir{num_imgs}(3,:) = 1;
        xiq{num_imgs}=zeros(3,nMatches);
        xiq{num_imgs}(1,:) = x_i_q(results(1,:),1)./tc;
        xiq{num_imgs}(2,:) = x_i_q(results(1,:),2)./tc;
        xiq{num_imgs}(3,:) = 1;
        original_dists(num_imgs) = size(dists,1);
        nidx = nidx+nMatches;
        N_img(num_imgs)=size(xiq{num_imgs},2);
        img_index(num_imgs)=r;
        %Increase the counter
        num_imgs = num_imgs+1;
        
        
        
        if show_figures
            % Create a new image showing the two images side by side.
            Ir = imread(reference_paths{r});
            im3 = appendimages(Iq,Ir);
            
            % Show a figure with lines joining the accepted matches.
            figure(1);
            %figure('Position', [100 100 size(im3,2) size(im3,1)]);
            colormap('gray');
            imagesc(im3);
            hold on;
            cols1 = size(Iq,2);
            for i = 1: size(results,2)
                line([coord_q(results(1,i),1) coord_r(results(2,i))+cols1], [coord_q(results(1,i),2) coord_r(results(2,i),2) ], 'Color', 'c');
            end
            hold off;
            nMatches
            pause;
        end
        
        
    end
    
end

num_imgs = validRefs;

%Generate the final arrays of data
x_i_q_t=zeros(2,nidx);
x_i_r_t=zeros(2,nidx);
x_i_index=zeros(1,nidx);
new_dists=zeros(nidx,1);

nidx=0;
for i=1:1:num_imgs
    nidx2=length(xiq{i}(1,:));
    x_i_q_t(:,nidx+1:nidx+nidx2)=xiq{i}(1:2,:);
    x_i_r_t(:,nidx+1:nidx+nidx2)=xir{i}(1:2,:);
    x_i_index(nidx+1:nidx+nidx2)=img_index(i)*ones(1,nidx2);
    new_dists(nidx+1:nidx+nidx2)=dists_i{i};
    nidx=nidx+nidx2;
end

%%%%%%%%%%%%%%%%%%%%%%
%%%  GENERATIVE MODEL  %%%
%%%%%%%%%%%%%%%%%%%%%%
disp(['Starting Generative Model  with ' num2str(num_imgs) ' images']);
% Lambda initialization
if vis_similarity
    lambda=zeros(K,num_imgs);
    for k=1:1:K
        for i=1:num_imgs
            lambda(k,i) = ext_initial_lambda;
        end
    end
end



% Figure representation (Total accepted points in query)
if show_figures
    
    Iqtmp3=Iq;
    for i=1:length(x_i_q_t)
        
        fc = round(x_i_q_t(1,i).*tc);
        sc = round(x_i_q_t(2,i).*tc);
        
        min_c_x=max(1,round(fc)-1);
        max_c_x=min(wc,round(fc)+1);
        min_c_y=max(1,round(sc)-1);
        max_c_y=min(hc,round(sc)+1);
        
        Iqtmp3(min_c_y:max_c_y,min_c_x:max_c_x,:)=0;
        Iqtmp3(min_c_y:max_c_y,min_c_x:max_c_x,1)=255;
    end
    
    figure(1),imshow(Iqtmp3);
    title('Total accepted points in query');
end

% New N
N = nidx;
mu=zeros(2,K);
sigma=zeros(2,2,K);

% ri initialization
if(K==2)
    initial_ri=ones(N,K);
    initial_ri(:,1)=0.10;
    initial_ri(:,2:K)=1./repmat(new_dists,1,K-1);
    initial_ri=initial_ri./repmat(sum(initial_ri,2),1,K);
    ri=initial_ri';
else
    initial_ri=rand(N,K);
    initial_ri(:,1)=0.1*multA;
    div_h=0.5*(hc/tc);
    div_w=(1/(K-1))*(wc/tc);
    medias=(div_w/2):div_w:wc/tc;
    for k=2:K
        mu(:,k)=[medias(k-1) div_h]';
        desv=(0.1*div_w)^2;
        sigma(:,:,k)=desv*eye(2);
        determinant=desv^2;
        if determinant<1e-10
            disp('prob3 determinant < 1e-10');
        end
        fact_pi=1/((2*pi)^(2/2));
        factor = fact_pi*(1/sqrt(determinant));
        
        for i=1:N
            tmp_vec1 = x_i_q_t(:,i)-mu(:,k);
            aux=tmp_vec1'/sigma(:,:,k);
            pSC(i,k) = max(factor.*exp((-1/2).*(aux*tmp_vec1)),1e-3);
        end
        initial_ri(:,k)=pSC(:,k);
    end
    
    initial_ri=initial_ri./repmat(sum(initial_ri,2),1,K);
    ri=initial_ri';
end

%Initialization of geometric Transforms
for img=1:num_imgs
    for k=2:K
        sigma_A{img}(:,:,k)=priorA;
        A{img}(:,:,k)=eye(3);
    end
end

%%% =========================== HM ITERATIONS ===========================

disp(['Processing... :: Iter 1/',num2str(niter)]);

for iter=1:niter
    
    %:::: Stop condition
    if iter>3
        mejora=(sum(logL(:,iter-1))-sum(logL(:,iter-2)))/sum(abs(logL(:,iter-2)));
        if mejora<stop_threshold
            iter=iter-1;
            disp(['Algorithm ended in ',num2str(iter),' iterations.'])
            if show_figures
                figure(3),plot(1:iter,sum(logL(:,1:iter)));
                show_result(x_i_q_t,ri,Iq);
            end
            break;
        end
    end
    
    %:::: Display current iter
    if mod(iter,10)==0
        disp(['Processing... :: Iter ',num2str(iter),'/',num2str(niter)]);
    end
    
    %:::: Clearing temporal variables
    clear mu;
    clear sigma;
    clear pSC;
    clear current_ri;
    clear pTL;
    
    %:::: UDATING PRIORS p_zi
    p_zi=(1/N)*sum(ri')';
    
    
    % :::: SPATIAL CONSISTENCY DISTRIBUTION
    %Allocating variables
    pSC=zeros(N,K);
    pTL=cell(1,num_imgs);
    priorL=zeros(1,num_imgs);
    fact_pi=1/((2*pi)^(2/2));
    
    %Using Gaussians
    if ~seg_SC
        
        %TODO: Falta mu, sigma por inicializar
        % :::: MU / SIGMA
        for k=2:K
            mu(:,k) = (sum((repmat(ri(k,:),2,1).*x_i_q_t)')')./sum(ri(k,:));
            sigma(:,:,k) = ((repmat(ri(k,:),2,1).*(x_i_q_t-repmat(mu(:,k),1,N)))*(x_i_q_t-repmat(mu(:,k),1,N))')./sum(ri(k,:));
        end
        
        % :::: PROB3: Gaussian(MU,SIGMA) for FG components
        fact_pi=1/((2*pi)^(2/2));
        for k=2:K
            
            determinant=det(sigma(:,:,k));
            
            % Fixing  determinant for degenerated cases
            if determinant<1e-06
                sigma(:,:,k)=sigma(:,:,k) + 1e-8*rand(2,2);
                determinant=det(sigma(:,:,k));
            end
            
            factor = fact_pi*(1/sqrt(determinant));
            
            for i=1:N
                tmp_vec1 = x_i_q_t(:,i)-mu(:,k);
                aux=tmp_vec1'/sigma(:,:,k);
                pSC(i,k) = max(factor.*exp((-1/2).*(aux*tmp_vec1)),minValue);
            end
            
        end
        % :::: PROB3: Uniform for BG component
        pSC(:,1) = mult*ones(N,1);
        %Using Previous Segmentations
    else
        cell_prob=zeros(as_idx,K);
        for k=1:K
            tot_ri_k = sum(ri(k,:)');
            for is=1:as_idx
                sum_ri_cell = 0;
                for iri=1:N
                    coordA=round(x_i_q_t(1,iri)*tc);
                    coordB=round(x_i_q_t(2,iri)*tc);
                    %if color_asign(coordA,coordB)==is
                    if color_asign(coordB,coordA)==is
                        sum_ri_cell = sum_ri_cell + ri(k,iri);
                    end
                end
                cell_prob(is,k) = sum_ri_cell/tot_ri_k;
                for iri=1:N
                    coordA=round(x_i_q_t(1,iri)*tc);
                    coordB=round(x_i_q_t(2,iri)*tc);
                    if color_asign(coordB,coordA)==is
                        pSC(iri,k)=cell_prob(is,k);
                    end
                end
            end
        end
        clear cell_prob;
    end
    
    
    
    
    % :::: PER IMAGE PROCESSING
    for img=1:num_imgs
        %Real index
        real_img = img_index(img);
        
        %Allocating variables
        pTL{img}=zeros(N_img(img),K);
        %         if seg_SC==0
        %             B{img}(:,k)
        
        % :::: TRANSFORMATION-BASED LOCATION Gaussian(A,SIGMA_A)
        for k=2:K
            current_ri=ri(k,find(x_i_index==real_img));
            
            % :::: Computing parameters
            %We check if we have points asigned to components to make
            %computations
            npoints=sum(current_ri>0.01);
            if(npoints>1)
                beta_aux=b_reg*length(current_ri)/K;
                aux=(repmat(current_ri,3,1).*(xir{img}))*(xir{img})';
                A{img}(:,:,k) = ((repmat(current_ri,3,1).*xiq{img})*(xir{img})')/aux;
                dif=xiq{img}-A{img}(:,:,k)*(xir{img});
                dif=dif(1:2,:);
                aux=((repmat(current_ri,2,1).*dif)*dif');
                sigma_A{img}(:,:,k) = (beta_aux*priorA+aux)/(beta_aux+sum(current_ri));
                determinantA=det(sigma_A{img}(:,:,k));
            else
                %Fixing degenerated case
                beta_aux=b_reg*length(current_ri)/K;
                sigma_A{img}(:,:,k)=priorA;
                determinantA=det(sigma_A{img}(:,:,k));
            end
            %Fixing degenerated case
            if(isnan(determinantA))
                sigma_A{img}(:,:,k)=priorA;% + 1e-3*rand(2,2);
                determinantA=det(sigma_A{img}(:,:,k));
            end
            
            % :::: Computing the distribution
            factor2 = fact_pi*(1/sqrt(determinantA));
            for i=1:N_img(img)
                tmp_vec2 = xiq{img}(:,i)-A{img}(:,:,k)*xir{img}(:,i);
                tmp_vec2=tmp_vec2(1:2);
                aux=tmp_vec2'/sigma_A{img}(:,:,k);
                pTL{img}(i,k) = max(factor2.*exp(-0.5*(aux*tmp_vec2)),minValue);
            end
            
            % :::: Prior L term (4 viene de 2*d => d=2 dimension de la
            % matriz)
            aux=beta_aux*priorA;
            aux=aux/sigma_A{img}(:,:,k);
            priorL(img)=priorL(img)-0.5*beta_aux*log(determinantA)-0.5*trace(aux);%
            
            % :::: In case we use a Gaussian distribution for the pSC, we
            % need a normalization factor
            if seg_SC==0
                Nr=size(xiq{img},2);
                Sigma_c=sigma_A{img}(1:2,1:2,k)+sigma(:,:,k);
                det_c=det(2*pi*Sigma_c);
                mu_aux=A{img}(:,:,k)*(xir{img});
                aux=repmat(mu(:,k),1,Nr)-mu_aux(1:2,:);
                aux1=aux'/Sigma_c;
                exponente=(aux1*aux);
                exponente=diag(exponente);
                B{img}(:,k)=(det_c^(-1/2))*exp(-0.5*exponente);
                if(sum(B{img}(:,k)==0))
                    disp('error B');
                    B{img}(:,k)=max(B{img}(:,k),minValue);
                end
            end
        end
        
        pTL{img}(:,1) = multA*ones(N_img(img),1);
        
        if(seg_SC==0)
            B{img}(:,1)=ones(N_img(img),1);
            B{img}=1./B{img};
        end
        
        
        
        % :::: NEW RI_K CALCULATION
        
        if seg_SC==0
            k=1;
            ri(k,find(x_i_index==real_img)) = p_zi(k).*B{img}(:,k).*pTL{img}(:,k).*pSC(find(x_i_index==real_img),k);
        else
            k=1;
            ri(k,find(x_i_index==real_img)) = p_zi(k).*pTL{img}(:,k).*pSC(find(x_i_index==real_img),k);
        end
        
        for k=2:K
            if seg_SC==0
                if vis_similarity==0
                    ri(k,find(x_i_index==real_img)) =  p_zi(k).*B{img}(:,k).*pTL{img}(:,k).*pSC(find(x_i_index==real_img),k);
                else
                    ri(k,find(x_i_index==real_img)) =  p_zi(k).*(lambda(k,img).*exp(-lambda(k,img).*dists_i{img}')).*B{img}(:,k).*pTL{img}(:,k).*pSC(find(x_i_index==real_img),k);
                end
            else
                if vis_similarity==0
                    ri(k,find(x_i_index==real_img)) =  p_zi(k).*pTL{img}(:,k).*pSC(find(x_i_index==real_img),k);
                else
                    ri(k,find(x_i_index==real_img)) =  p_zi(k).*(lambda(k,img).*exp(-lambda(k,img).*dists_i{img}')).*pTL{img}(:,k).*pSC(find(x_i_index==real_img),k);
                end
            end
        end
        
        
        % :::: RI NORMALIZATION
        suma=sum(ri(:,find(x_i_index==real_img)));
        for k=1:K
            ri(k,find(x_i_index==real_img))=ri(k,find(x_i_index==real_img))./suma;
        end
        
        
        
        
        % :::: LOGL CALCULATION
        tmp_L = zeros(N,1);
        
        
        
        %Computing Likelihooods
        for k=1:K
            tmp_indx = find(x_i_index==real_img);
            for i=1:length(tmp_indx);
                ix = tmp_indx(i);
                if ri(k,ix)<1e-100 || isnan(ri(k,ix))
                    %We do not do anything
                else
                    if seg_SC==0
                        if vis_similarity==0 || k==1
                            tmp_L(i) = tmp_L(i) + ri(k,ix)'.*log((( p_zi(k).*B{img}(i,k).*pTL{img}(i,k).*pSC(ix,k)))./ri(k,ix)');
                        else
                            tmp_L(i) = tmp_L(i) + ri(k,ix)'.*log((( p_zi(k).*(lambda(k,img).*exp(-lambda(k,img).*dists_i{img}(i))).*B{img}(i,k).*pTL{img}(i,k).*pSC(ix,k)))./ri(k,ix)');
                        end
                    else
                        if vis_similarity==0 || k==1
                            tmp_L(i) = tmp_L(i) + ri(k,ix)'.*log((( p_zi(k).*pTL{img}(i,k).*pSC(ix,k)))./ri(k,ix)');
                        else
                            tmp_L(i) = tmp_L(i) + ri(k,ix)'.*log((( p_zi(k).*(lambda(k,img).*exp(-lambda(k,img).*dists_i{img}(i))).*pTL{img}(i,k).*pSC(ix,k)))./ri(k,ix)');
                        end
                    end
                    
                end
            end
        end
        
        
        logL(img,iter) = sum(tmp_L) + priorL(img);
        
        if (isnan(logL(img,iter)) || isinf(logL(img,iter)))
            disp(['NaN: log L :: non-convergence :: ',num2str(iter),' iters']);
            hm_error=1;
            return
        end
        
    end % End -per image processing-
    
    
    % :::: LAMBDA ADAPTATION
    if ext_constant_lambda==0
        for k=2:1:K
            for img=1:num_imgs
                real_img = img_index(img);
                current_ri=ri(k,find(x_i_index==real_img));
                %current_ri=ri(k,:);
                if(sum(current_ri>0))
                    lambda(k,img) = sum(current_ri)/(current_ri*dists_i{img}(:));
                else
                    lambda(k,img) = ext_initial_lambda;
                end
            end
        end
    end

    if show_figures
        if(mod(iter,10)==0)
            figure(3),plot(1:iter,sum(logL(:,1:iter)));
            show_result(x_i_q_t,ri,Iq);
        end
    end
    
    
    
    % :::: RI RANKING
    similarities=zeros(Nref,1);
    for img = 1:num_imgs
        real_img = img_index(img) ;
        currt_ri=ri(:,x_i_index==real_img);
        similarities(real_img,1)=sum(sum(currt_ri(2:K,:)))/original_dists(img);
    end
end
    % End -hm iterations-
    
    if show_figures
        for img = 1:num_imgs
            real_img = img_index(img) ;
            currt_ri=ri(:,x_i_index==real_img);
            reference_paths(real_img)
            show_result(x_i_q_t(:,x_i_index==real_img),currt_ri,Iq);
        end
    end
%%% =========================== SEGMENTATION ===========================

function [region_borders,mult,as_idx,color_asign,Iq_segmt] = segment(Iq,ext3_sigma,ext3_k,ext3_min,segPath)

imwrite(Iq,'tempquery.ppm','ppm');
unix([segPath ' ' num2str(ext3_sigma),' ',num2str(ext3_k),' ',num2str(ext3_min),' tempquery.ppm tempsegout.ppm']);

Iq_segmt = imread('tempsegout.ppm');

as_idx=0;

% Color assignment

color_asign = zeros(size(Iq_segmt,1),size(Iq_segmt,2));

for as_i=1:size(Iq_segmt,1)
    for as_j=1:size(Iq_segmt,2)
        
        if as_idx==0
            as_idx=1;
            existingcolors(as_idx,1)=Iq_segmt(as_i,as_j,1);
            existingcolors(as_idx,2)=Iq_segmt(as_i,as_j,2);
            existingcolors(as_idx,3)=Iq_segmt(as_i,as_j,3);
            existingcolors(as_idx,4)=as_idx;
            
            color_asign(as_i,as_j)=as_idx;
        else
            for as_lp=1:as_idx
                as_found=0;
                
                if Iq_segmt(as_i,as_j,1)==existingcolors(as_lp,1) && Iq_segmt(as_i,as_j,2)==existingcolors(as_lp,2) && Iq_segmt(as_i,as_j,3)==existingcolors(as_lp,3)
                    color_asign(as_i,as_j)=existingcolors(as_lp,4);
                    as_found=1;
                    break;
                end
            end
            
            if as_found==0
                as_idx = as_idx+1;
                existingcolors(as_idx,1)=Iq_segmt(as_i,as_j,1);
                existingcolors(as_idx,2)=Iq_segmt(as_i,as_j,2);
                existingcolors(as_idx,3)=Iq_segmt(as_i,as_j,3);
                existingcolors(as_idx,4)=as_idx;
                
                color_asign(as_i,as_j)=as_idx;
            end
        end
    end
end

clear existingcolors;

mult=1/as_idx;

region_borders = imdilate(Iq_segmt,ones(3,3)) > imerode(Iq_segmt,ones(3,3));


%%% =========================== SHOW RESULTS ===========================

function show_result(x_i_q_t,ri,Iq)

Iq_final=Iq;
[hc wc aux]=size(Iq);
tc=1024;
g=1;

for i=1:length(ri)
    
    k = find(ri(:,i)==max(ri(:,i)));
    
    if k>1
        
        coordx=round(x_i_q_t(1,i)*tc);
        coordy=round(x_i_q_t(2,i)*tc);
        
        Iq_final(coordy-g:coordy+g,coordx-g:coordx+g,:)=0;
        Iq_final(coordy-g:coordy+g,coordx-g:coordx+g,k-1)=255;
        
    end
end

figure(2),imshow(Iq_final);
title ('Current HM points in query');

disp('Done. Press any key to continue.')
pause;

