%% Peeling decoder evolution for Protograph-based LDPC codes

% Both LDPC and SC-LDPC
% Multiple Connections allowed
% Puncturing
% Initialization in Matlab, rest in C (Mex file)

% This software compute the expected graph evolution for a protograph-based
% LDPC code defined by a base-matrix (Bconv) and a puncturing vector (p_mask).
% We illustrate the use for a $(3,6)$-regular LDPC block code and for 
% a SC-LDPC code based on the $(3,6)$-regular LDPC block code.

% The script can be use to evaluate the scaling parameter gamma and the
% variance parameter delta_1^*. To estimate the latter, flagVarD1 has to be
% set to 1. WARNING: estimating delta_1^* can be very slow.

% Before running the script, the mex-file has to be compilated: >> mex PD_LDPC.cpp

% Do not hesitate to contact me for any question.

% January 2015, Pablo Martinez Olmos olmos@tsc.uc3m.es, Markus Stinner markus.stinner@tum.de


clear all;
close all;
clc;


%% %%%%%%%%%%%%%%%%%%%%%%%%%%% INITIALIZE  %%%%%%%%%%%%%%%%%%%%%%%%%%%
epsilon=0.42;                   % BEC erasure probability
N=500 ;                          % N^-1 is the step size we use in the numerical 
                                % evaluation of the solution of the differential equations
                                % The bigger it is, the more precise the
                                % solution will be and the slower the
                                % simulation will be (N>1 !!).                               

flagVarD1=0;                    % If 1, we compute the variance approximation 
                                % WARNING: estimating delta_1^* can be very slow.



savefile=[ 'SET_A_FILENAME' '.mat'];

%%%%%%%%%%%%%%%%%%  Uncoupled (3,6)-regular LDPC code %%%%%%%%%%%%%%%%%%%%%%%


B=[3 3];
[nc,nv]=size(B);
B0=[];
Bconv=B;
L=1;                %Uncoupled
ms=0;
p_mask=ones(1,nv);  %Puncturing mask: zero for puncturing the corresponding bit in the base matrix



%%%%%%%%%%%%%%%%%%  SC-LDPC code based on (3,6)-regular LDPC code %%%%%%%%%%%%%%%%%%%%%%%

% B=[3,3];
% [bc,bv]=size(B);
% L=50;               %Chain length
% ms=2;
% 
% l=3; %choose l!     %Variable degree
% 
% B0=[1 1];
% nc=(L+ms)*bc;
% nv=L*bv;
% 
% Bconv=zeros(nc,nv);
% 
% for j=0:l-1
%     for cL=1:L
%         Bconv( (cL-1+j)*bc+1:(cL+j)*bc , (cL-1)*bv+1:cL*bv )=B0;
%     end %for i
% end %for j
% 
% p_mask=ones(1,nv);   %Puncturing mask: zero for puncturing the corresponding bit in the base matrix





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% BEYOND THIS POINT YOU DO NOT HAVE TO CONFIGURE ANYTHING %%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



%% %%%%%%%%%%%%%%%%%% BEC Transmission and PD initialization  %%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Transmission...');
[ T,R,Reps,Veps,Initial_check ] = BEC_transmission( Bconv, p_mask, epsilon );



%% %%%%%%%%%%%%%%%%%%%%%%%%%%% PD Preparation  %%%%%%%%%%%%%%%%%%%%%%%%%%%%
format long;

eps_mean=mean(ones(1,nv).*(1-p_mask)+p_mask*epsilon);

numIt=N*nv*eps_mean;             % Number of PD iterations allowed
norm_time=N;                            % Normalized time factor
Rmat=zeros(size(R,1),numIt);            % check DD after each PD
Vmat=zeros(nv,numIt);           % node DD after each PD
VarD1mat=zeros(1,numIt);                % approximation to the Variance

Rmat(:,1)=Reps;
Vmat(:,1)=Veps;

d_c=sum(Bconv,1);
SD1=[-1*max(d_c):1:max(d_c)-1];         % Vector of all possible changes in the total number c1  of deg1 check nodes
%PD1=zeros(size(SD1));                  % At each PD iteration, here we store the probabilities of each possible change in c1


%% %%%%%%%%%%%%%%%%%%%%%%%%%%% PD Analysis  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Step 0: Decodable patterns. Easy in LDPC. We identify what patterns in T
%are decodable. This only has to be computed once.
D=sum(T,2)==1;

% CALL OF THE MEX FUNCTION
[~,b]=unique(Initial_check);
counts_check=max(diff([0;b]));          % number of check types each original check type generates after BEC transmission

only=1;

% compute expected finishing time
disp('Test iteration...');
tic;
[~,~,~,~,~,~,]=PD_LDPC(N,Bconv,nc,nv,L,T+zeros(size(T)),Initial_check,1,2,norm_time,Rmat(:,1),Vmat(:,1),D+zeros(size(D)),size(R,1),ms,max(sum(T>0,1)),sum(T>0,1)',counts_check,1,SD1,max(d_c),flagVarD1,only);
T0=toc;

fprintf('The simulation is expected to finish in %f minutes.\n You can speed up the simulation (losing accuracy) by setting a smaller N value. \n',T0*numIt/60);

prompt='Do you want to continue (1==y(es) / 0==n(o))?';
answer= input(prompt);
if (answer~=1)
    return
else
    
    % start analysis
    tic;
        [RPD,VPD,it,VARD,MEAND,evol_var,evol_check]=PD_LDPC(N,Bconv,nc,nv,L,T+zeros(size(T)),Initial_check,1,numIt,norm_time,Rmat(:,1),Vmat(:,1),D+zeros(size(D)),size(R,1),ms,max(sum(T>0,1)),sum(T>0,1)',counts_check,1,SD1,max(d_c),flagVarD1,only);
    Tfinal=toc;
    
    fprintf('Final Simulation time: %f minutes \n',Tfinal/60);    
end %if

time=(0:1:it-1)/N/(nv/L);
c1=sum(RPD(D==1,1:1:it)/(nv/L),1);
var=sum(VPD(1:1:it),1);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% PLOT RESULTS                                            %%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% Output data:
%
% time: normalize time vector (tau)
% c1: expected evolution of the degree-one check nodes in the graph
% var: expected evolution of the fraction of variable nodes in the graph
%

figure(1);
plot(time,c1,'b','Linewidth',2);
xlabel('$\epsilon$','Interpreter','Latex','Fontsize',20);
ylabel('Expected evolution $\mathtt{E}[c_1(\tau)]$','Interpreter','Latex','Fontsize',15);
hold on;

figure(2);
plot(time,var,'b','Linewidth',2);
xlabel('$\epsilon$','Interpreter','Latex','Fontsize',20);
ylabel('Expected evolution  $\mathtt{E}[v(\tau)]$','Interpreter','Latex','Fontsize',15);

save(savefile,'time','c1','var');



